import React, { useState, useEffect } from "react";
import "./App.css";
import {
  StarPrinterFactory,
  InterfaceType,
  StarConnectionSettings,
  Options,
  StarXpandCommand,
  StarIO10Logger,
} from "star-io10-web";

function Printing_Template() {
  const [fieldDataIndex, setFieldDataIndex] = useState<number>(0);
  const [identifier, setIdentifier] = useState<string>("");
  const [printerFactory, setPrinterFactory] = useState<StarPrinterFactory>(
    () => {
      console.log("create printer factory");
      return new StarPrinterFactory();
    },
  );
  const handleBeforeUnloadEvent = (event: BeforeUnloadEvent) => {
    showMessage("dispose printer factory");
    printerFactory.dispose();
  };
  useEffect(() => {
    // when the window is closed, dispose the printer factory.
    window.addEventListener("beforeunload", handleBeforeUnloadEvent, true);
    return () =>
      window.removeEventListener("beforeunload", handleBeforeUnloadEvent, true);
  }, []);

  const [isProcessing, setIsProcessing] = useState<boolean>(false);
  const [message, setMessage] = useState<string>("");
  async function showMessage(msg: string) {
    console.log(msg);
    setMessage((prev: string) => prev + msg + "\n");
  }

  async function onPressPrintButton() {
    showMessage("on press print button");
    setIsProcessing(true);

    // Start debug log output to console
    // StarIO10Logger.start();

    const connectionSettings = new StarConnectionSettings();
    connectionSettings.interfaceType = InterfaceType.Usb;
    if (identifier.length > 0) {
      connectionSettings.identifier = identifier;
    }

    const printer = printerFactory.createStarPrinter(connectionSettings);
    // const printer = new StarPrinter(connectionSettings);
    // `new StarPrinter(connectionSettings)` may take a long time to process.
    // If you want to separate the process to create StarPrinter, use StarPrinterFactory.

    try {
      await printer.open();
      showMessage("open success");

      let fieldData = "";
      if (fieldDataIndex === 0) {
        fieldData = FIELD_DATA_0;
      }
      if (fieldDataIndex === 1) {
        fieldData = FIELD_DATA_1;
      }
      if (fieldDataIndex === 2) {
        fieldData = FIELD_DATA_2;
      }

      const options = new Options.PrintOptions();
      options.template = await createTemplate();
      await printer.print(fieldData, options);

      showMessage("print success");

      await printer.close();
      showMessage("close success");
    } catch (error) {
      try {
        await printer?.close();
      } catch {
        // do nothing.
      }

      showMessage("print failed " + error);
      console.error(error);
    } finally {
      // StarIO10Logger.stop();
      await printer.dispose();
      setIsProcessing(false);
    }
  }

  async function createTemplate(): Promise<string> {
    const builder = new StarXpandCommand.StarXpandCommandBuilder();

    builder.addDocument(
      new StarXpandCommand.DocumentBuilder()
        .settingPrintableArea(48.0)
        .addPrinter(
          new StarXpandCommand.PrinterBuilder()
            .actionPrintImage(
              new StarXpandCommand.Printer.ImageParameter("logo_01.png", 406),
            )
            .styleInternationalCharacter(
              StarXpandCommand.Printer.InternationalCharacterType.Usa,
            )
            .styleCharacterSpace(0.0)
            .add(
              new StarXpandCommand.PrinterBuilder()
                .styleAlignment(StarXpandCommand.Printer.Alignment.Center)
                .styleBold(true)
                .styleInvert(true)
                .styleMagnification(
                  new StarXpandCommand.MagnificationParameter(2, 2),
                )
                .actionPrintText("${store_name}\n"),
            )
            .actionFeed(1.0)
            .actionPrintText(
              "Order ${order_number}",
              new StarXpandCommand.Printer.TextParameter().setWidth(16),
            )
            .actionPrintText(" ")
            .actionPrintText(
              "${time}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                15,
                new StarXpandCommand.Printer.TextWidthParameter()
                  .setAlignment(StarXpandCommand.Printer.TextAlignment.Right)
                  .setEllipsizeType(
                    StarXpandCommand.Printer.TextEllipsizeType.End,
                  ),
              ),
            )
            .actionPrintText(
              "Sale for ${sales_type}",
              new StarXpandCommand.Printer.TextParameter().setWidth(16),
            )
            .actionPrintText(" ")
            .actionPrintText(
              "Served by ${server}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                15,
                new StarXpandCommand.Printer.TextWidthParameter()
                  .setAlignment(StarXpandCommand.Printer.TextAlignment.Right)
                  .setEllipsizeType(
                    StarXpandCommand.Printer.TextEllipsizeType.End,
                  ),
              ),
            )
            .actionPrintText("Transaction #${transaction_id}\n")
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .add(
              new StarXpandCommand.PrinterBuilder(
                new StarXpandCommand.Printer.PrinterParameter().setTemplateExtension(
                  new StarXpandCommand.TemplateExtensionParameter().setEnableArrayFieldData(
                    true,
                  ),
                ),
              )
                .actionPrintText(
                  "${item_list.quantity}",
                  new StarXpandCommand.Printer.TextParameter().setWidth(2),
                )
                .actionPrintText(
                  "${item_list.name}",
                  new StarXpandCommand.Printer.TextParameter().setWidth(24),
                )
                .actionPrintText(
                  "${item_list.unit_price%6.2lf}\n",
                  new StarXpandCommand.Printer.TextParameter().setWidth(6),
                ),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionPrintText(
              "Subtotal",
              new StarXpandCommand.Printer.TextParameter().setWidth(26),
            )
            .actionPrintText(
              "${subtotal%6.2lf}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                6,
                new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                  StarXpandCommand.Printer.TextEllipsizeType.End,
                ),
              ),
            )
            .actionPrintText(
              "Tax",
              new StarXpandCommand.Printer.TextParameter().setWidth(26),
            )
            .actionPrintText(
              "${tax%6.2lf}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                6,
                new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                  StarXpandCommand.Printer.TextEllipsizeType.End,
                ),
              ),
            )
            .add(
              new StarXpandCommand.PrinterBuilder()
                .styleBold(true)
                .actionPrintText(
                  "Total",
                  new StarXpandCommand.Printer.TextParameter().setWidth(26),
                )
                .actionPrintText(
                  "${total%6.2lf}\n",
                  new StarXpandCommand.Printer.TextParameter().setWidth(
                    6,
                    new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                      StarXpandCommand.Printer.TextEllipsizeType.End,
                    ),
                  ),
                ),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionPrintText(
              "${credit_card_number}",
              new StarXpandCommand.Printer.TextParameter().setWidth(26),
            )
            .actionPrintText(
              "${total%6.2lf}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                6,
                new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                  StarXpandCommand.Printer.TextEllipsizeType.End,
                ),
              ),
            )
            .actionPrintText(
              "Approval Code",
              new StarXpandCommand.Printer.TextParameter().setWidth(16),
            )
            .actionPrintText(
              "${approval_code}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                16,
                new StarXpandCommand.Printer.TextWidthParameter().setAlignment(
                  StarXpandCommand.Printer.TextAlignment.Right,
                ),
              ),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionPrintText(
              "Amount",
              new StarXpandCommand.Printer.TextParameter().setWidth(26),
            )
            .actionPrintText(
              "${amount%6.2lf}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                6,
                new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                  StarXpandCommand.Printer.TextEllipsizeType.End,
                ),
              ),
            )
            .actionPrintText(
              "Total",
              new StarXpandCommand.Printer.TextParameter().setWidth(26),
            )
            .actionPrintText(
              "${total%6.2lf}\n",
              new StarXpandCommand.Printer.TextParameter().setWidth(
                6,
                new StarXpandCommand.Printer.TextWidthParameter().setEllipsizeType(
                  StarXpandCommand.Printer.TextEllipsizeType.End,
                ),
              ),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionPrintText("Signature\n")
            .add(
              new StarXpandCommand.PrinterBuilder()
                .styleAlignment(StarXpandCommand.Printer.Alignment.Center)
                .actionPrintImage(
                  new StarXpandCommand.Printer.ImageParameter(
                    "signature.png",
                    256,
                  ),
                ),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(32.0).setX(8.0),
            )
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionPrintText("\n")
            .styleAlignment(StarXpandCommand.Printer.Alignment.Center)
            .actionPrintText("${address}\n")
            .actionPrintText("${tel}\n")
            .actionPrintText("${mail}\n")
            .actionFeed(1.0)
            .actionPrintText("${url}\n")
            .actionPrintRuledLine(
              new StarXpandCommand.Printer.RuledLineParameter(48.0),
            )
            .actionFeed(2.0)
            .actionPrintText("Powered by Star Micronics\n")
            .actionPrintBarcode(
              new StarXpandCommand.Printer.BarcodeParameter(
                "${transaction_id}",
                StarXpandCommand.Printer.BarcodeSymbology.Code128,
              ).setPrintHri(true),
            )
            .actionCut(StarXpandCommand.Printer.CutType.Partial),
        ),
    );

    const commands = await builder.getCommands();
    return Promise.resolve(commands);
  }

  const FIELD_DATA_0 = `
    {
        "store_name" : "   Star Cafe   ",
        "order_number" : "#2-007",
        "time" : "10/16 11:13PM",
        "sales_type" : "Walk-in",
        "server" : "Jane",
        "transaction_id" : "0123456789",
        "item_list" : [
            {
                "name" : "Vanilla Latte",
                "unit_price" : 4.99,
                "quantity" : 1
            },
            {
                "name" : "Chocolate Chip Cookie",
                "unit_price" : 3.25,
                "quantity" : 1
            }
        ],
        "subtotal" : 8.24,
        "tax" : 0.73,
        "total" : 8.97,
        "credit_card_number" : "VISA 0123",
        "approval_code" : "OK2443",
        "amount" : 8.97,
        "address" : "123 Star Road, City,\\nState 12345",
        "tel" : "123-456-7890",
        "mail" : "info@star-m.jp",
        "url" : "star-m.jp"
    }`;

  const FIELD_DATA_1 = `
    {
        "store_name" : "   Star Cafe   ",
        "order_number" : "#2-008",
        "time" : "02/09 12:40PM",
        "sales_type" : "Takeout",
        "server" : "Jessica",
        "transaction_id" : "0111111111",
        "item_list" : [
            {
                "name" : "Vanilla Latte with caramel sauce",
                "unit_price" : 5.99,
                "quantity" : 1
            },
            {
                "name" : "Star's Homemade Chocolate Chip Cookie",
                "unit_price" : 4.99,
                "quantity" : 1
            }
        ],
        "subtotal" : 10.98,
        "tax" : 0.97,
        "total" : 11.95,
        "credit_card_number" : "VISA 0456",
        "approval_code" : "OK5667",
        "amount" : 11.95,
        "address" : "123 Star Road, City,\\nState 12345",
        "tel" : "123-456-7890",
        "mail" : "info@star-m.jp",
        "url" : "star-m.jp"
    }`;

  const FIELD_DATA_2 = `
    {
        "store_name" : "   Star Cafe   ",
        "order_number" : "#2-009",
        "time" : "05/23 10:10AM",
        "sales_type" : "Takeout",
        "server" : "James",
        "transaction_id" : "0222222222",
        "item_list" : [
            {
                "name" : "Chai Tea Latte",
                "unit_price" : 5.49,
                "quantity" : 1
            },
            {
                "name" : "Americano",
                "unit_price" : 2.99,
                "quantity" : 3
            },
            {
                "name" : "Orange Juice",
                "unit_price" : 3.99,
                "quantity" : 1
            },
            {
                "name" : "Shortbread Cookies",
                "unit_price" : 3.49,
                "quantity" : 2
            },
            {
                "name" : "Chocolate Brownie",
                "unit_price" : 4.49,
                "quantity" : 3
            }
        ],
        "subtotal" : 38.9,
        "tax" : 3.45,
        "total" : 42.35,
        "credit_card_number" : "VISA 0789",
        "approval_code" : "OK7889",
        "amount" : 42.35,
        "address" : "123 Star Road, City,\\nState 12345",
        "tel" : "123-456-7890",
        "mail" : "info@star-m.jp",
        "url" : "star-m.jp"
    }`;

  return (
    <div className="Printing_Template">
      {isProcessing && (
        <div className="App-overlay">
          <div className="App-overlay-message">Printing...</div>
        </div>
      )}
      <div>
        <label>Identifier: </label>
        <input
          value={identifier}
          onChange={(e) => setIdentifier(e.target.value)}
        />
        <br />
        <label> FieldData: </label>
        <select
          value={fieldDataIndex}
          onChange={(e) => setFieldDataIndex(Number(e.target.value))}
        >
          <option value={0}> Receipt 1</option>
          <option value={1}> Receipt 2</option>
          <option value={2}> Receipt 3</option>
        </select>
        <br />
        <button id="print" onClick={onPressPrintButton}>
          Print
        </button>
        <br />
        <div style={{ whiteSpace: "pre-line" }}>
          <p>{message}</p>
        </div>
      </div>
    </div>
  );
}

export default Printing_Template;
